/*--------------------------------------------------------------------*
 *
 * Developed by;
 *	Neal Horman - http://www.wanlink.com
 *	Copyright (c) 2003 Neal Horman. All Rights Reserved
 *
 *	Redistribution and use in source and binary forms, with or without
 *	modification, are permitted provided that the following conditions
 *	are met:
 *	1. Redistributions of source code must retain the above copyright
 *	   notice, this list of conditions and the following disclaimer.
 *	2. Redistributions in binary form must reproduce the above copyright
 *	   notice, this list of conditions and the following disclaimer in the
 *	   documentation and/or other materials provided with the distribution.
 *	3. All advertising materials mentioning features or use of this software
 *	   must display the following acknowledgement:
 *	This product includes software developed by Neal Horman.
 *	4. Neither the name Neal Horman nor the names of any contributors
 *	   may be used to endorse or promote products derived from this software
 *	   without specific prior written permission.
 *	
 *	THIS SOFTWARE IS PROVIDED BY NEAL HORMAN AND ANY CONTRIBUTORS ``AS IS'' AND
 *	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *	ARE DISCLAIMED.  IN NO EVENT SHALL NEAL HORMAN OR ANY CONTRIBUTORS BE LIABLE
 *	FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 *	DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *	OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 *	LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *	OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *	SUCH DAMAGE.
 *
 *	CVSID:  $Id: dnsbl.c,v 1.22 2004/11/26 00:20:55 neal Exp $
 *
 * DESCRIPTION:
 *	application:	spamilter
 *	module:		dnsbl.c
 *--------------------------------------------------------------------*/

static char const cvsid[] = "@(#)$Id: dnsbl.c,v 1.22 2004/11/26 00:20:55 neal Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>

#include "spamilter.h"
#include "dns.h"
#include "dnsbl.h"
#include "misc.h"
#include "smisc.h"

void dnsbl_open(mlfiPriv *priv, char *dbpath)
{	char		*str;
	char		buf[8192];
	char		numbuf[128];
	int		fd;
	RBLHOST		*prbl;

	if(priv != NULL)
	{
		asprintf(&str,"%s/db.rdnsbl",dbpath);
		fd = open(str,O_RDONLY);
		if(fd == -1)
			mlfi_debug("dnsbl_open: unable to open RDNSBL host file '%s'\n",str);
		free(str);

		priv->dnsrblhostqty = 0;

		if(fd != -1)
		{
			lseek(fd,0l,SEEK_SET);
			while(mlfi_fdgets(fd,buf,sizeof(buf)) >= 0)
			{
				/* handle comments */
				str = strchr(buf,'#');
				if(str != NULL)
				{
					/* truncate at comment */
					*(str--) = '\0';
					/* trim right */
					while(str >= buf && (*str ==' ' || *str == '\t'))
						*(str--) = '\0';
				}

				if(strlen(buf))
				{
					priv->dnsrblhosts = realloc(priv->dnsrblhosts,sizeof(RBLHOST)*(priv->dnsrblhostqty+1));
					prbl = priv->dnsrblhosts + priv->dnsrblhostqty;
					priv->dnsrblhostqty++;

					memset(prbl,0,sizeof(RBLHOST));
					str = mlfi_strcpyadv(prbl->hostname,sizeof(prbl->hostname),buf,'|');
					str = mlfi_strcpyadv(prbl->url,sizeof(prbl->url),str,'|');
					str = mlfi_strcpyadv(numbuf,sizeof(numbuf),str,'|');
					prbl->action =
						strcasecmp(numbuf,"tag") == 0 ? RBL_A_TAG :
						strcasecmp(numbuf,"reject") == 0 ? RBL_A_REJECT :
						RBL_A_NULL;
					str = mlfi_strcpyadv(numbuf,sizeof(numbuf),str,'|');
					prbl->stage =
						strcasecmp(numbuf,"conn") == 0 ? RBL_S_CONN :
						strcasecmp(numbuf,"from") == 0 ? RBL_S_FROM :
						strcasecmp(numbuf,"rcpt") == 0 ? RBL_S_RCPT :
						strcasecmp(numbuf,"eom") == 0 ? RBL_S_EOM :
						RBL_S_NULL;
				}
			}
			close(fd);
		}
	}
}

void dnsbl_close(mlfiPriv *priv)
{
	if(priv != NULL)
	{
		free(priv->dnsrblmatch);
		priv->dnsrblmatch = NULL;

		free(priv->dnsrblhosts);
		priv->dnsrblhosts = NULL;
	}
}

int isLocalIp(struct sockaddr *ipin)
{	int	rc = 0;

	/*
	 *	On the assumption that these networks are filtered by the
	 *	gateway router to this network, these then are assumed to
	 *	be local ip's and therefore not in need of checking against
	 *	any rdnsbl.
	 */

	/* erm.. this is ip4 only right now */
	if(ipin != NULL && ipin->sa_family == AF_INET)
	{	long	ip = ntohl(((struct sockaddr_in *)ipin)->sin_addr.s_addr);

		/* always allow localhost */
		rc |= ((ip&0xff000000l) == 0x7f000000l);	/* ip = 127.0.0.0/8 */

		/* rfc1918 networks */
		rc |= ((ip&0xff000000l) == 0x0a000000l);	/* ip = 10.0.0.0/8 */
		rc |= ((ip&0xfff00000l) == 0xac100000l);	/* ip = 172.16.0.0/12 */
		rc |= ((ip&0xffff0000l) == 0xc0a80000l);	/* ip = 192.168.0.0/16 */

		/* draft-manning-dsua-03.txt (1 May 2000) nets, specifically DHCP auto-configuration */
		rc |= ((ip&0xffff0000l) == 0xa9fe0000l);	/* ip = 169.254.0.0/16 */
	}

	if(rc)
		mlfi_debug("Localip\n");

	return(rc);
}

RBLHOST *dnsbl_action(RBLHOST **prbls, int stage)
{	RBLHOST	*prblh = NULL;

	while(prblh == NULL && prbls != NULL && *prbls != NULL)
	{
		if((*prbls)->stage == stage && (*prbls)->action > RBL_A_NULL)
			prblh = *prbls;
		else
			prbls++;
	}
	mlfi_debug("dnsbl_action: %s %s%s\n",stage == RBL_S_CONN ? "connect" : stage == RBL_S_RCPT ? "rcpt" : stage == RBL_S_FROM ? "from" : "?",
		prblh == NULL ? "no action" : prblh->hostname,
		prblh == NULL ? "" : prblh->action == RBL_A_TAG ? " Tag" : prblh->action == RBL_A_REJECT ? " Reject" : " Unknown"
		);

	return(prblh);
}

void dnsbl_check(mlfiPriv *priv)
{	int		i;
	RBLHOST		**prbl;
	RBLHOST		*prblh;

	if(priv != NULL && priv->ip.sa_family == AF_INET && !isLocalIp(&priv->ip) && priv->dnsrblmatch == NULL)
	{
		prbl = priv->dnsrblmatch = (RBLHOST **) calloc(priv->dnsrblhostqty+1,sizeof(RBLHOST *));
		if(priv->dnsrblhosts != NULL)
		{
			for(i=0; i<priv->dnsrblhostqty; i++)
			{
				prblh = priv->dnsrblhosts + i;
				if(dns_rdnsbl_has_rr_a(priv->statp,ntohl(((struct sockaddr_in *)&priv->ip)->sin_addr.s_addr), prblh->hostname))
				{
					mlfi_debug("\t\t'%s.%s' - Blacklisted\n",priv->ipstr,prblh->hostname);
					*prbl = prblh;
					prbl++;
					priv->dnsblcount++;
				}
			}
		}
		else
			mlfi_debug("dnsbl_check: unable to alloc rdnsbl host list\n");
	}
}

void dnsbl_add_hdr(SMFICTX *ctx, RBLHOST *prbl)
{	mlfiPriv	*priv = MLFIPRIV;

	if(priv != NULL && prbl != NULL)
		mlfi_addhdr_printf(ctx,"X-RDNSBL-Warning","Source IP tagged as spam source by %s",prbl->hostname);
}

/*--------------------------------------------------------------------*
 * $Log: dnsbl.c,v $
 * Revision 1.22  2004/11/26 00:20:55  neal
 * neal - 041125 - add res_n support
 *
 * Revision 1.21  2004/11/05 03:31:27  neal
 * neal - 041104 - fix memory leak. leadin to ip6 compatibility
 *
 * Revision 1.20  2004/04/10 01:37:08  neal
 * neal - 040409 - make it slightly more readable and used enums - no functional changes
 *
 * Revision 1.19  2003/10/04 16:16:35  neal
 * neal - 031004 - fix for reading blank lines
 *
 * Revision 1.18  2003/09/06 19:02:10  neal
 * neal - 030906 - mod ipfwmtad nolonger runs unless uid == 0. mod spamilter logging to record reject reason in field 5. add ipfwmtad cli operators for addition and removal queuing of ipaddress to and from the fwblock list. mod spamilter all extended smtp error codes are now 5.7.1
 *
 * Revision 1.17  2003/08/10 21:52:59  neal
 * neal - 030810 - don\'t cary the sendmail context around to places that only need the private context
 *
 * Revision 1.16  2003/05/28 03:37:57  neal
 * neal - 030527 - include license in all files
 *
 * Revision 1.15  2003/04/14 04:02:07  neal
 * neal - 030413 - fix dnsbl_check
 *
 * Revision 1.14  2003/04/14 03:52:07  neal
 * neal - 030413 - move the rdnsbl hosts to a flat ascii file
 *
 * Revision 1.13  2003/04/07 02:29:32  neal
 * neal - 030406 - add code to dnsblupd to check for success/failure by doing a lookup following the update request.
 *
 * Revision 1.12  2003/04/03 03:13:53  neal
 * neal - 030402 - move MTA hostname check into default sndr action so that a BWL_A_xxxx will work as expected
 *
 * Revision 1.11  2003/03/31 05:49:59  neal
 * neal - 030330 - do not link the milter to the bind resolver library, it segfaults (and is is huge!)
 *
 * Revision 1.10  2003/03/30 05:03:37  neal
 * neal - 030329 - use multihop.dsbl.org and unconfirmed.dsbl.org as advisory only.
 *
 * Revision 1.9  2003/03/24 02:23:09  neal
 * neal - 030323 - bad pointer fixes, etc
 *
 * Revision 1.8  2003/03/24 00:46:26  neal
 * neal - 030323 - add realtime black/white list of sender/recipient email addresses
 *
 * Revision 1.7  2003/03/20 06:42:24  neal
 * neal - 030320 - add accept/reject/spam logging. make "Sender address deliverable" check work for local domain.
 *
 * Revision 1.6  2003/03/02 04:11:49  neal
 * neal - 030301 - add smtp sender deliverablity check
 *
 * Revision 1.5  2003/02/24 20:45:53  neal
 * neal - 030224 - add dns blacklist cli util
 *
 * Revision 1.4  2003/02/24 08:14:47  neal
 * neal - 030224 - add some debug junk
 *
 * Revision 1.3  2003/02/24 05:04:42  neal
 * neal - 030223 - remove individual dnsbl zone tracking
 *
 * Revision 1.2  2003/02/24 04:11:33  neal
 * neal - 030223 - add header/footers to all files
 *
 *--------------------------------------------------------------------*/
