/*--------------------------------------------------------------------*
 *
 * Developed by;
 *	Neal Horman - http://www.wanlink.com
 *	Copyright (c) 2003 Neal Horman. All Rights Reserved
 *
 *	Redistribution and use in source and binary forms, with or without
 *	modification, are permitted provided that the following conditions
 *	are met:
 *	1. Redistributions of source code must retain the above copyright
 *	   notice, this list of conditions and the following disclaimer.
 *	2. Redistributions in binary form must reproduce the above copyright
 *	   notice, this list of conditions and the following disclaimer in the
 *	   documentation and/or other materials provided with the distribution.
 *	3. All advertising materials mentioning features or use of this software
 *	   must display the following acknowledgement:
 *	This product includes software developed by Neal Horman.
 *	4. Neither the name Neal Horman nor the names of any contributors
 *	   may be used to endorse or promote products derived from this software
 *	   without specific prior written permission.
 *	
 *	THIS SOFTWARE IS PROVIDED BY NEAL HORMAN AND ANY CONTRIBUTORS ``AS IS'' AND
 *	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *	ARE DISCLAIMED.  IN NO EVENT SHALL NEAL HORMAN OR ANY CONTRIBUTORS BE LIABLE
 *	FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 *	DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *	OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 *	LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *	OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *	SUCH DAMAGE.
 *
 *	CVSID:  $Id: hndlrs.c,v 1.81 2005/07/18 00:55:13 neal Exp $
 *
 * DESCRIPTION:
 *	application:	spamilter
 *	module:		hndlrs.c
 *--------------------------------------------------------------------*/

static char const cvsid[] = "@(#)$Id: hndlrs.c,v 1.81 2005/07/18 00:55:13 neal Exp $";

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <sys/wait.h>
#include <resolv.h>
#include <ctype.h>

#include "spamilter.h"
#include "hndlrs.h"
#include "smtp.h"
#include "dns.h"
#include "dnsbl.h"
#include "misc.h"
#include "bwlist.h"
#include "inet.h"
#include "badext.h"
#include "smisc.h"
#include "ifi.h"
#ifdef SUPPORT_POPAUTH
#include "popauth.h"
#endif
#ifdef SUPPORT_LIBSPF
#include "spfapi.h"
#endif

#ifndef INET_ADDRSTRLEN
#define INET_ADDRSTRLEN 16
#endif

#ifndef strcasestr
/* This function was wholesale copied from FreeBSD's libc/string/strcasestr.c
   see the file strcasestr.lic for license and copyright information
*/ 
char *
strcasestr(s, find)
        register const char *s, *find;
{
        register char c, sc;
        register size_t len;

        if ((c = *find++) != 0) {
                c = tolower((unsigned char)c);
                len = strlen(find);
                do {
                        do {
                                if ((sc = *s++) == 0)
                                        return (NULL);
                        } while ((char)tolower((unsigned char)sc) != c);
                } while (strncasecmp(s, find, len) != 0);
                s--;
        }
        return ((char *)s);
}
#endif

char *str2lo(char *src)
{	char *p = src;

	while(p != NULL && *p)
	{
		*p = (char)tolower(*p);
		p++;
	}

	return(src);
}

void log_status(mlfiPriv *priv, char *status, char *reason)
{
	if(priv != NULL && ((struct sockaddr_in*)&priv->ip)->sin_addr.s_addr != INADDR_ANY)
	{
		if(reason != NULL && priv->rcpt != NULL && priv->sndr != NULL)
			syslog(LOG_INFO,"%s %s %s %s %s",status,priv->ipstr,priv->sndr,priv->rcpt,reason);

		else if(priv->subject != NULL && priv->rcpt != NULL && priv->sndr != NULL)
			syslog(LOG_INFO,"%s %s %s %s %s",status,priv->ipstr,priv->sndr,priv->rcpt,priv->subject);

		else if(priv->rcpt != NULL && priv->sndr != NULL)
			syslog(LOG_INFO,"%s %s %s %s",status,priv->ipstr,priv->sndr,priv->rcpt);

		else if(priv->sndr != NULL)
			syslog(LOG_INFO,"%s %s %s",status,priv->ipstr,priv->sndr);

		else 
			syslog(LOG_INFO,"%s %s",status,priv->ipstr);
	}
}

void log_reject(mlfiPriv *priv, char *reason)
{
	log_status(priv,"Rejected",reason);
}

void log_accept(mlfiPriv *priv)
{
	log_status(priv,"Accepted",NULL);
}

sfsistat mlfi_rdnsbl_reject(SMFICTX *ctx, sfsistat *rs, int stage)
{	mlfiPriv	*priv = MLFIPRIV;
	RBLHOST		*prblh = NULL;
	char		*reason;

	if(*rs == SMFIS_CONTINUE && 
		priv->dnsblcount &&
		(prblh = dnsbl_action(priv->dnsrblmatch,stage)) != NULL &&
		prblh->action == RBL_A_REJECT)
	{
		mlfi_setreply(ctx,550,"5.7.1","Rejecting due to security policy - DNSBL, Please see: %s#dnsbl and %s",gPolicyUrl,prblh->url);
		mlfi_debug("DNSBL rejected - %s\n",priv->ipstr);
		asprintf(&reason,"DNSBL '%s'",prblh->url);
		log_reject(priv,reason);
		free(reason);
		*rs = SMFIS_REJECT;
	}

	return(*rs);
}

sfsistat mlfi_hndlr_exec(SMFICTX *ctx, int stage)
{	mlfiPriv	*priv = MLFIPRIV;
	sfsistat	rs = SMFIS_CONTINUE;
	char		*str = NULL;
	char		*reason = NULL;
	int		rc;

	if(priv != NULL)
	{
		switch(stage)
		{
			case RBL_S_FROM:
				asprintf(&str,"%s '%s' '%s'",priv->sndractionexec,priv->ipstr,priv->sndr);
				break;
			case RBL_S_RCPT:
				asprintf(&str,"%s '%s' '%s' '%s'",priv->rcptactionexec,priv->ipstr,priv->sndr,priv->rcpt);
				break;
		}

		if(str != NULL)
		{
			mlfi_debug("mlfi_hndlr_exec: '%s'\n",str);
			rc = system(str);
			rc = WIFEXITED(rc) ? WEXITSTATUS(rc) : 2;
			mlfi_debug("mlfi_hndlr_exec: returned %d\n",rc);
			asprintf(&reason,"Exec '%s'",str);
			switch(rc)
			{
				case 1:
					rs = SMFIS_REJECT;
					mlfi_setreply(ctx,550,"5.7.1","Rejecting due to security policy. Please see: %s#general",gPolicyUrl);
					log_reject(priv,reason);
					break;
				case 127:
					/* deliberate fall thru to case 2 */
				case 2:
					rs = SMFIS_TEMPFAIL;
					log_status(priv,"TempFailed",reason);
					break;
				case 3:
					rs = SMFIS_DISCARD;
					log_status(priv,"Discarded",reason);
					break;
				default:
					/* deliberate fall thru to case 0 */
				case 0:
					rs = SMFIS_CONTINUE;
					break;
			}
			free(str);
			free(reason);
		}
	}

	return(rs);
}

void mlfi_MtaHostIpfwAction(char *ipstr, char *action)
{	int sd = NetSockOpenTcpPeer(INADDR_LOOPBACK,4739);

	if(sd != INVALID_SOCKET)
	{	char	buf[1024];

		while(NetSockGets(sd,buf,sizeof(buf)-1,1) > 0)
			;
		NetSockPrintf(sd,"%s,%s\r\n\r\n",action,ipstr);
		NetSockClose(&sd);
	}
}

sfsistat mlfi_connect(SMFICTX *ctx, char *hostname, _SOCK_ADDR *hostaddr)
{	mlfiPriv	*priv = MLFIPRIV;
	sfsistat	rs = SMFIS_CONTINUE;

	/* don't assume that we need to create a context structure */
	if(priv == NULL)
	{
		priv = calloc(1,sizeof(mlfiPriv));
		smfi_setpriv(ctx, priv);
		bwlist_init(priv);
		priv->statp = RES_NALLOC(priv->statp);
		if(priv->statp != NULL)
			res_ninit(priv->statp);
		badext_init(priv,gDbpath);
	}

	if (priv != NULL)
	{
		/* make sure that the b/w list action files are open */
		bwlist_open(priv,gDbpath);
		if (gDnsBlChk)
			dnsbl_open(priv,gDbpath);

		memset(&priv->ip,0,sizeof(priv->ip));
		if(hostaddr != NULL)
 		{	char	buf[INET_ADDRSTRLEN];

			memcpy(&priv->ip,hostaddr,sizeof(priv->ip));

			memset(buf,0,sizeof(buf));
			inet_ntop(((struct sockaddr_in *)&priv->ip)->sin_family,&((struct sockaddr_in *)&priv->ip)->sin_addr,buf,sizeof(buf));
			priv->ipstr = strdup(buf);
		}

		if(hostname != NULL)
			priv->iphostname = strdup(hostname);

		mlfi_debug("\nconnect: '%s'/%s\n",hostname,priv->ipstr);


		priv->islocalnethost = (
			ntohl(((struct sockaddr_in *)&priv->ip)->sin_addr.s_addr) == INADDR_LOOPBACK
			|| ifi_islocalnet(ntohl(((struct sockaddr_in *)&priv->ip)->sin_addr.s_addr))
#ifdef SUPPORT_POPAUTH
			|| (gPopAuthChk != NULL && popauth_validate(priv,gPopAuthChk))
#endif
			);

		if(gDnsBlChk && !priv->islocalnethost)
			dnsbl_check(priv);
		mlfi_rdnsbl_reject(ctx,&rs,RBL_S_CONN);

		if(gMtaHostIpfwNominate && !priv->islocalnethost)
			mlfi_MtaHostIpfwAction(priv->ipstr,"inculpate");
	}

	return(rs);
}

sfsistat mlfi_helo(SMFICTX *ctx, char *helohost)
{	mlfiPriv *priv = MLFIPRIV;
	sfsistat rs = SMFIS_CONTINUE;

	mlfi_debug("helo: '%s'\n",helohost);

	if(priv != NULL)
		priv->helo = str2lo(strdup(helohost));

	return(rs);
}

sfsistat mlfi_envfrom(SMFICTX *ctx, char **envfrom)
{	mlfiPriv	*priv = MLFIPRIV;
	sfsistat	rs = SMFIS_CONTINUE;

	mlfi_debug("envfrom: '%s'\n",*envfrom);

	if(priv != NULL)
	{	char	*frm = str2lo(strdup(*envfrom));
		char	*dom = NULL;

		if(priv->sndr != NULL)
			free(priv->sndr);
		priv->sndr = strdup(frm);
		mboxdomainsplit(frm,&dom);
		priv->sndraction = bwlist_query_action(priv,BWL_L_SNDR,dom,frm,priv->sndractionexec);

		switch(priv->sndraction)
		{
			case BWL_A_REJECT:
			case BWL_A_DISCARD:
			case BWL_A_TEMPFAIL:
			case BWL_A_ACCEPT:
			case BWL_A_TARPIT:
			case BWL_A_EXEC:
				/* take no further action */
				break;
			default:
				/* deliberate fall thru to BWL_A_NULL case */
			case BWL_A_NULL:
				/* no acton specified, do other normal checks */
				if(priv->helo == NULL || !*priv->helo)
				{
					mlfi_setreply(ctx,550,"5.7.1","Rejecting due to security policy - Helo or Ehlo MTA greeting expected, Please see: %s#helo",gPolicyUrl);
					mlfi_debug("envfrom: MTA hostname not specified\n");
					log_reject(priv,"Missing Helo/Ehlo");
					rs = SMFIS_REJECT;
				}
				break;
		}
		free(frm);
	}

	return(rs);
}

sfsistat mlfi_envrcpt(SMFICTX *ctx, char **argv)
{	mlfiPriv	*priv = MLFIPRIV;
	sfsistat	rs = SMFIS_CONTINUE;
	char		*reason = NULL;

	mlfi_debug("envrcpt: '%s'\n",*argv);

	if (priv != NULL)
	{	char	*rcpt = str2lo(strdup(*argv));
		char	*dom = NULL;

		if(priv->rcpt != NULL)
			free(priv->rcpt);
		priv->rcpt = strdup(rcpt);

		mboxdomainsplit(rcpt,&dom);
		priv->rcptaction = bwlist_query_action(priv,BWL_L_RCPT,dom,rcpt,priv->rcptactionexec);
		asprintf(&reason,"Blacklisted recipient '%s@%s'",rcpt,dom);

		switch(priv->rcptaction)
		{
			case BWL_A_TARPIT:
				sleep(120);	/* slow the bastard down! */
				/* deliberate fall thru to BWL_A_REJECT */
			case BWL_A_REJECT:
				mlfi_setreply(ctx,550,"5.7.1","Rejecting due to security policy - Recipient has been blacklisted, Please see: %s#blacklistedrecipient",gPolicyUrl);
				mlfi_debug("envrcpt: Blacklisted recipient '%s@%s'\n",rcpt,dom);
				log_reject(priv,reason);
				rs = SMFIS_REJECT;
				break;
			case BWL_A_TEMPFAIL:
				log_status(priv,"TempFailed",reason);
				rs = SMFIS_TEMPFAIL;
				break;
			case BWL_A_DISCARD:
				log_status(priv,"Discarded",reason);
				rs = SMFIS_DISCARD;
				break;
			case BWL_A_ACCEPT:
				/* take no further action */
				break;
			case BWL_A_EXEC:
				rs = mlfi_hndlr_exec(ctx,RBL_S_RCPT);
				break;
			default:
				priv->rcptaction = BWL_A_NULL;
				/* deliberate fall thru to BWL_A_NULL case */
			case BWL_A_NULL:
				/* no acton specified, do other normal checks */
				if(priv->sndraction != BWL_A_ACCEPT)
					mlfi_rdnsbl_reject(ctx,&rs,RBL_S_RCPT);
				break;
		}

		free(reason);
		asprintf(&reason,"Blacklisted Sender '%s'",priv->sndr);

		if(rs == SMFIS_CONTINUE && priv->rcptaction == BWL_A_NULL)
		{
			/* this is done here instead of in envfrom so that the recpient can be collected for logging purposes */
			switch(priv->sndraction)
			{
				case BWL_A_TARPIT:
					sleep(120);	/* slow the bastard down! */
					/* deliberate fall thru to BWL_A_REJECT */
				case BWL_A_REJECT:
					mlfi_setreply(ctx,550,"5.7.1","Rejecting due to security policy - Sender '%s' has been blacklisted, Please see: %s#blacklistedsender",priv->sndr,gPolicyUrl);
					mlfi_debug("envrcpt: Blacklisted SMTP Sender, '%s'\n",priv->sndr);
					log_reject(priv,reason);
					rs = SMFIS_REJECT;
					break;
				case BWL_A_TEMPFAIL:
					log_status(priv,"TempFailed",reason);
					rs = SMFIS_TEMPFAIL;
					break;
				case BWL_A_DISCARD:
					log_status(priv,"Discarded",reason);
					rs = SMFIS_DISCARD;
					break;
				case BWL_A_ACCEPT:
					/* take no further action */
					break;
				case BWL_A_EXEC:
					rs = mlfi_hndlr_exec(ctx,RBL_S_FROM);
					break;
				default:
					priv->sndraction = BWL_A_NULL;
					/* deliberate fall thru to BWL_A_NULL case */
				case BWL_A_NULL:
					if(gMtaHostChk && priv->helo != NULL && inet_addr(priv->helo) != -1)
					{
						mlfi_setreply(ctx,550,"5.7.1","Rejecting due to security policy - Ip address used as MTA hostname '%s', Please see: %s#ipusedashostname",priv->helo,gPolicyUrl);
						mlfi_debug("envrcpt: Ip address used as MTA hostname '%s'\n",priv->helo);
						free(reason);
						asprintf(&reason,"Ip address used as MTA hostname '%s'",priv->helo);
						log_reject(priv,reason);
						rs = SMFIS_REJECT;

						if(gMtaHostIpfw)
							mlfi_MtaHostIpfwAction(priv->ipstr,"add");
					}
					else if(gMtaHostChk && !priv->islocalnethost && priv->helo != NULL && dom != NULL && strcmp(priv->helo,dom) == 0)
					{
						mlfi_setreply(ctx,550,"5.7.1","Rejecting due to security policy - Invalid source hostname '%s', Please see: %s#invalidsourcehostname",priv->helo,gPolicyUrl);
						mlfi_debug("envrcpt: Invalid source MTA hostname '%s'\n",priv->helo);
						free(reason);
						asprintf(&reason,"Invalid source MTA hostname '%s'",priv->helo);
						log_reject(priv,reason);
						rs = SMFIS_REJECT;

						if(gMtaHostIpfw)
							mlfi_MtaHostIpfwAction(priv->ipstr,"add");
					}
					else if(gMtaHostChk && !priv->islocalnethost && !dns_query_rr_a(priv->statp,priv->helo))
					{
						mlfi_setreply(ctx,550,"5.7.1","Rejecting due to security policy - Invalid hostname '%s', Please see: %s#invalidhostname",priv->helo,gPolicyUrl);
						mlfi_debug("envrcpt: Invalid MTA hostname '%s'\n",priv->helo);
						free(reason);
						asprintf(&reason,"Invalid MTA hostname '%s'",priv->helo);
						log_reject(priv,reason);
						rs = SMFIS_REJECT;

						if(gMtaHostIpfw)
							mlfi_MtaHostIpfwAction(priv->ipstr,"add");
					}
					else if(gMtaHostIpChk &&
						!priv->islocalnethost &&
						strcmp(priv->helo,priv->iphostname) != 0 &&
						!dns_hostname_ip_match(priv->statp,priv->helo,ntohl(((struct sockaddr_in *)&priv->ip)->sin_addr.s_addr))
						)
					{
						mlfi_setreply(ctx,550,"5.7.1","Rejecting due to security policy - Helo hostname/ip mismatch, Please see: %s#hostnameipmismatch",gPolicyUrl);
						mlfi_debug("envrcpt: Helo hostname/ip mismatch\n");
						free(reason);
						asprintf(&reason,"Hostname/ip mismatch '%s'",priv->helo);
						log_reject(priv,reason);
						rs = SMFIS_REJECT;
					}
#ifdef SUPPORT_LIBSPF
					else if(gMtaSpfChk && !priv->islocalnethost && mlfi_spf_reject(priv,&rs) == SMFIS_REJECT)
					{
						mlfi_setreply(ctx,550,"5.7.1","Rejecting due to security policy - SPF %s - %s",priv->spf_rs,priv->spf_explain);
						mlfi_debug("envrcpt: SPF %s: '%s'\n",priv->spf_rs,priv->spf_error);
						free(reason);
						asprintf(&reason,"SPF %s: '%s'",priv->spf_rs,priv->spf_error);
						log_reject(priv,reason);
						rs = SMFIS_REJECT;
					}
#endif
					else if(mlfi_rdnsbl_reject(ctx,&rs,RBL_S_FROM) == SMFIS_CONTINUE &&
						gSmtpSndrChk &&
						!priv->islocalnethost &&
						ntohl(((struct sockaddr_in *)&priv->ip)->sin_addr.s_addr) != INADDR_LOOPBACK &&
						strcmp("<>",priv->sndr) != 0
						)
					{	RBLHOST		**prbls = NULL;
						char		*frm = strdup(priv->sndr);
						char		*dom = NULL;

						mboxdomainsplit(frm,&dom);

						for(priv->sndrreject = 0,prbls=priv->dnsrblmatch; prbls != NULL && *prbls != NULL; prbls++)
							priv->sndrreject |= ((*prbls)->action == RBL_A_REJECT);

						priv->smtprc = -1;
						priv->SmtpSndrChkFail = (!priv->sndrreject && !smtp_email_address_is_deliverable(priv->statp,frm,dom,&priv->smtprc));

						if(priv->SmtpSndrChkFail && strcasecmp(gSmtpSndrChkAction,"Reject") == 0)
						{
							if(priv->smtprc == -1 || priv->smtprc/100 == 4)
							{
								if(priv->smtprc == -1)
									priv->smtprc = 450;
								mlfi_setreply(ctx,priv->smtprc,"4.7.1","Temporary failure - Unable to validate Sender address %s, Please see: %s#tempfailinvalidsender",priv->sndr,gPolicyUrl);
								log_status(priv,"TempFailed","Sender address verification");
								rs = SMFIS_TEMPFAIL;
							}
							else
							{
								mlfi_setreply(ctx,550,"5.7.1","Rejecting due to security policy - Unable to validate Sender address %s, Please see: %s#invalidsender",priv->sndr,gPolicyUrl);
								log_reject(priv,"Sender address verification");
								rs = SMFIS_REJECT;
							}
							mlfi_debug("envrcpt: Return path '%s' not deliverable\n",priv->sndr);
						}
						free(frm);
					}
					break;
			}
		}

		free(reason);

		priv->reject = (rs != SMFIS_CONTINUE);
		free(rcpt);
	}

	switch(rs)
	{
		case SMFIS_REJECT:
			mlfi_debug("envrcpt: Reject\n");
			break;
		case SMFIS_DISCARD:
			mlfi_debug("envrcpt: Discard\n");
			break;
		case SMFIS_TEMPFAIL:
			mlfi_debug("envrcpt: Fail\n");
			break;
		default:
			break;
	}

	return(rs);
}

sfsistat mlfi_header(SMFICTX *ctx, char *headerf, char *headerv)
{	mlfiPriv *priv = MLFIPRIV;
	sfsistat rs = SMFIS_CONTINUE;
	char	*attachfname = NULL;
	char	*reason;

	if (priv != NULL)
	{
		if(strcasecmp(headerf,"X-Status") == 0 && strlen(headerv))
			priv->xstatus = strdup(headerv);
		if(strcasecmp(headerf,"Subject") == 0)
			priv->subject = strdup(headerv);

		priv->MsExtFound = (gMsExtChk &&
			(mlfi_findBadExtHeader(priv->badextlist,priv->badextqty,headerf,"Content-Disposition:","filename=",headerv,&attachfname) ||
			(attachfname == NULL && mlfi_findBadExtHeader(priv->badextlist,priv->badextqty,headerf,"Content-Type:","name=",headerv,&attachfname)))
			);

		if(priv->MsExtFound && strcasecmp(gMsExtChkAction,"Reject") == 0)
		{
			mlfi_debug("header: Attachement Reject\n");
			rs = SMFIS_REJECT;
			mlfi_setreply(ctx,550,"5.7.1","Rejecting due to security policy - Inline executable attachement, Please see: %s#attachmentinline",gPolicyUrl);
			asprintf(&reason,"Attachement name '%s'",attachfname);
			log_reject(priv,reason);
			free(reason);
		}

		if(attachfname != NULL)
			free(attachfname);
	}

	return(rs);
}

sfsistat mlfi_body(SMFICTX *ctx, u_char *bodyp, size_t bodylen)
{	mlfiPriv *priv = MLFIPRIV;
	sfsistat rs = SMFIS_CONTINUE;

	/* collect the body */
	if (priv != NULL)
	{	char	*p = (char *)realloc(priv->body, priv->bodylen + bodylen + 1);

		if(p != NULL)
		{
			priv->body = p;
			memcpy((priv->body+priv->bodylen), bodyp, bodylen);
			priv->bodylen += bodylen;
			priv->body[priv->bodylen] = '\0';
		};
	}

	return(rs);
}

int mlfi_prependSubjectHeader(SMFICTX *ctx, char *fmt, ...)
{	mlfiPriv	*priv = MLFIPRIV;
	int		rc = 0;
	char		*str = NULL;
	va_list		vl;

	if(ctx != NULL && priv != NULL && fmt != NULL)
	{
		va_start(vl,fmt);
		rc = vasprintf(&str,fmt,vl);
		va_end(vl);

		if(str != NULL)
		{
			if(priv->subject != NULL)
			{	char	*s;

				asprintf(&s,"%s - %s",str,priv->subject);
				if(s != NULL)
				{
					smfi_chgheader(ctx,"Subject",1,s);
					free(s);
				}
			}
			else
				smfi_addheader(ctx,"Subject",str);
			free(str);
		}
		else
			rc = -1;
	}

	return(rc);
}

sfsistat mlfi_eom(SMFICTX *ctx)
{	mlfiPriv	*priv = MLFIPRIV;
	sfsistat	rs = SMFIS_CONTINUE;
	RBLHOST		**prbls = NULL;
	char		*attachfname = NULL;
	char		*reason = NULL;
	int		tag = 0;

	if (priv != NULL)
	{
		if(priv->sndraction != BWL_A_ACCEPT && priv->rcptaction != BWL_A_ACCEPT)
		{
			priv->MsExtFound |= (gMsExtChk && priv->body != NULL && mlfi_findBadExtBody(priv->badextlist,priv->badextqty,priv->body,&attachfname));

			if(priv->MsExtFound && strcasecmp(gMsExtChkAction,"Reject") == 0)
			{
				mlfi_debug("eom: Attachement Reject\n");
				mlfi_setreply(ctx,550,"5.7.1","Rejecting due to security policy - Executable attachement, Please see: %s#attachmentbody",gPolicyUrl);
				asprintf(&reason,"Attachement name '%s'",attachfname);
				log_reject(priv,reason);
				free(reason);
				rs = SMFIS_REJECT;
			}
			else if(priv->MsExtFound && strcasecmp(gMsExtChkAction,"Tag") == 0)
			{
				mlfi_prependSubjectHeader(ctx,"Virus ?");

				mlfi_debug("eom: Flagged - Virus\n");
				log_status(priv,"Virus",NULL);
				tag = 1;
			}
			else if(priv->SmtpSndrChkFail && strcasecmp(gSmtpSndrChkAction,"Tag") == 0)
			{
				mlfi_prependSubjectHeader(ctx,"Valid Sender ?");

				mlfi_debug("eom: Flagged - Invalid Sender\n");
				log_status(priv,"Sender",NULL);
				tag = 1;
			}
			else if(priv->dnsblcount > 0)
			{
				mlfi_addhdr_printf(ctx,"X-RDNSBL-Count","%d",priv->dnsblcount);
				for(prbls=priv->dnsrblmatch; prbls != NULL && *prbls != NULL; prbls++)
				{
					if((*prbls)->stage == RBL_S_EOM && (*prbls)->action == RBL_A_TAG)
						dnsbl_add_hdr(ctx,*prbls);
				}

				mlfi_prependSubjectHeader(ctx,"Spam ?");

				mlfi_debug("eom: Flagged - Spam\n");
				log_status(priv,"Spam",NULL);
				tag = 1;
			}
			else
			{
				mlfi_debug("eom: Accepted\n");
				log_accept(priv);
			}
#ifdef SUPPORT_LIBSPF
			if(gMtaSpfChk && priv->spf_rs != NULL)
			{
				mlfi_addhdr_printf(ctx,"X-Spamilter-SPF", "%s (%s) reciever=%s; client-ip=%s; envelope-from=%s; helo=%s;)",
					priv->spf_rs,priv->spf_error,gHostname,priv->ipstr,priv->sndr,priv->helo);
			}
#endif
			if(tag)
			{
				if(priv->xstatus != NULL)
					smfi_chgheader(ctx,"X-Status",1,"F");
				else
					smfi_addheader(ctx,"X-Status","F");

				smfi_addheader(ctx,"X-Keywords","$Label5");
			}

			if(attachfname != NULL)
				free(attachfname);
		}
		else
		{
			mlfi_debug("eom: Accepted\n");
			log_accept(priv);
		}
	}

	mlfi_addhdr_printf(ctx,"X-Milter","%s (Reciever: %s; Sender-ip: %s; Sender-helo: %s;)",mlfi.xxfi_name,gHostname,priv->ipstr,priv->helo);

	if(rs == SMFIS_CONTINUE && gMtaHostIpfwNominate && !priv->islocalnethost)
		mlfi_MtaHostIpfwAction(priv->ipstr,"exculpate");

	mlfi_cleanup(ctx);

	return(rs);
}

sfsistat mlfi_abort(SMFICTX *ctx)
{	sfsistat rs = mlfi_cleanup(ctx);

	return(rs);
}

sfsistat mlfi_close(SMFICTX *ctx)
{	mlfiPriv *priv = MLFIPRIV;
	sfsistat rs;

	rs = mlfi_cleanup(ctx);

	if(priv != NULL)
	{
		if(priv->helo != NULL)
			free(priv->helo);
		if(priv->iphostname != NULL)
			free(priv->iphostname);
		if(priv->ipstr != NULL)
			free(priv->ipstr);
		if(priv->statp != NULL)
		{
			res_nclose(priv->statp);
			free(priv->statp);
		}
		bwlist_close(priv);
		dnsbl_close(priv);
		badext_close(priv);
		free(priv);
	}
	smfi_setpriv(ctx, NULL);

	return(rs);
}

sfsistat mlfi_cleanup(SMFICTX *ctx)
{	mlfiPriv *priv = MLFIPRIV;
	sfsistat rs = SMFIS_CONTINUE;

	if(priv != NULL)
	{
		if(priv->xstatus != NULL)
		{
			free(priv->xstatus);
			priv->xstatus = NULL;
		}
		if(priv->sndr != NULL)
		{
			free(priv->sndr);
			priv->sndr = NULL;
		}
		if(priv->rcpt != NULL)
		{
			free(priv->rcpt);
			priv->rcpt = NULL;
		}
		if(priv->subject != NULL)
		{
			free(priv->subject);
			priv->subject = NULL;
		}
		if(priv->body != NULL)
		{
			free(priv->body);
			priv->body = NULL;
		}
#ifdef SUPPORT_LIBSPF
		if(priv->spf_rs != NULL)
		{
			free(priv->spf_rs);
			priv->spf_rs = NULL;
		}
		if(priv->spf_error != NULL)
		{
			free(priv->spf_error);
			priv->spf_error = NULL;
		}
		if(priv->spf_explain != NULL)
		{
			free(priv->spf_explain);
			priv->spf_explain = NULL;
		}
#endif
	}

	return(rs);
}

/*--------------------------------------------------------------------*
 * $Log: hndlrs.c,v $
 * Revision 1.81  2005/07/18 00:55:13  neal
 * neal - 050717 - Wall cleanup
 *
 * Revision 1.80  2004/12/12 22:14:16  neal
 * neal - 041212 - add support for reading MsExtChk file list from db.extensions in lue of being hard coded
 *
 * Revision 1.79  2004/11/28 01:32:37  neal
 * neal - 041127 - Luns says that not having res_nclose will leak file descriptors, so put it back in.
 *
 * Revision 1.78  2004/11/26 06:42:39  neal
 * neal - 041126 - res_?close is not supposed to be public, so do not use it
 *
 * Revision 1.77  2004/11/26 05:09:41  neal
 * neal - 041125 - fix for solaris
 *
 * Revision 1.76  2004/11/26 04:25:08  neal
 * neal - 041125 - finish initial res_n changes
 *
 * Revision 1.75  2004/11/26 00:26:08  neal
 * neal - 041125 - add res_n support
 *
 * Revision 1.74  2004/11/05 05:13:36  neal
 * neal - 041104 - add first step towards ip6 support
 * add compile time support for libspf and popauth support
 *
 * Revision 1.73  2004/11/05 03:55:53  neal
 * neal - 041104 - add support for client authentication
 *
 * Revision 1.72  2004/04/22 05:04:57  neal
 * neal - 040421 - actually use the configured db path instead of the hard coded one - patch from Luns Tee
 * fix - make sure the priv->smtprc is initialized. - patch from Luns Tee
 *
 * Revision 1.71  2004/04/10 06:03:12  neal
 * neal - 040410 - crashland SPF. change X-Milter header to provide more information
 *
 * Revision 1.70  2004/04/10 01:39:08  neal
 * neal - 040409 - fix error url anchor syntax error
 *
 * Revision 1.69  2004/03/30 18:42:01  neal
 * neal - 040330 - fold the result of the popauth check into islocalnethost status. commonize code to do mtaipfwd client connections. add ip address as mta hostname check, and recipient domain name as mta hostname check, both based on MtaHostChk config option
 *
 * Revision 1.68  2004/03/19 04:14:49  neal
 * neal - 040318 - cache the result of ifi_islocalhost instead of calling it a bazillion times
 *
 * Revision 1.67  2004/03/19 03:20:50  neal
 * neal - 040318 - add support for pop-before-smtp or popauth
 *
 * Revision 1.66  2004/01/27 22:24:58  neal
 * neal - 040127 - fix blacklisted error printf specifier
 *
 * Revision 1.65  2004/01/04 01:32:51  neal
 * neal - 040103 - remove gMtaUrl hyperlink support in favor of gPolicyUrl and anchors
 *
 * Revision 1.64  2004/01/04 01:08:49  neal
 * neal - 040103 - more smtp error message updates, and a policy.html to match
 *
 * Revision 1.63  2004/01/02 20:18:23  neal
 * neal - 040102 - do not do mtahostchk or smtpsndrchk on local network hosts
 *
 * Revision 1.62  2004/01/02 17:51:08  neal
 * neal - 040102 - add global config.h include
 *
 * Revision 1.61  2004/01/02 17:41:07  neal
 * neal - 040102 - make smtp reply messages more consistent and add html anchor name. use ifi_islocalnet in favor of ifi_islocaip && islocalip combination
 *
 * Revision 1.60  2003/12/27 01:00:00  neal
 * neal - 031226 - fix off by one malloc
 *
 * Revision 1.59  2003/11/26 16:36:30  neal
 * neal - 031126 - dont add local ip addresses via ipfwmatd
 *
 * Revision 1.58  2003/11/04 02:50:05  neal
 * neal - 031103 - add exculpate operator, and internally rename nominate to inculpate
 *
 * Revision 1.57  2003/10/29 02:08:44  neal
 * neal - 031028 - less memory intensive filename extension checking
 *
 * Revision 1.56  2003/10/28 06:31:36  neal
 * neal - 031028 - fix inet_ntoa segfault on solaris
 *
 * Revision 1.55  2003/10/28 05:32:09  neal
 * neal - 031027 - ok, really handle white space in file attachment names
 *
 * Revision 1.54  2003/10/25 18:16:11  neal
 * neal - 031025 - handle white space in Content-Disposition and Content-Type headers, free attachment file name pointers
 *
 * Revision 1.53  2003/10/12 01:42:02  neal
 * neal - 031011 - do not nominate local ip addresses
 *
 * Revision 1.52  2003/10/04 16:23:31  neal
 * neal - 031004 - remove bogus private ipa member
 * 	add support for MtaHostIpfwNominate
 * 	add support for MtaHostIpChk
 * 	white listing a sender as "ACCEPT" now correctly does not do rdnsbl checks
 * 	don't free the reject "reason" pointer more than once per allocation.
 *
 * Revision 1.51  2003/09/20 17:21:57  neal
 * neal - 030920 - fix pointer type assigment warnings
 *
 * Revision 1.50  2003/09/19 04:55:28  neal
 * * fix -  don't free the reject "reason" pointer more that once per allocation. ;)
 * * add - global SmtpSndrChkAction and MsExtChkAction with 'Tag' or 'Reject' actions in
 * 	spamilter.rc to optionally tag a given email instead of rejecting it.
 *
 * Revision 1.49  2003/09/06 19:02:10  neal
 * neal - 030906 - mod ipfwmtad nolonger runs unless uid == 0. mod spamilter logging to record reject reason in field 5. add ipfwmtad cli operators for addition and removal queuing of ipaddress to and from the fwblock list. mod spamilter all extended smtp error codes are now 5.7.1
 *
 * Revision 1.48  2003/09/02 16:37:20  neal
 * neal - 030903 - handle -1 timeout return from mx testing
 *
 * Revision 1.47  2003/09/02 07:12:10  neal
 * neal - 030902 - fix segfaults. add MtaHostIpfw filter
 *
 * Revision 1.46  2003/08/24 18:09:35  neal
 * neal - 030824 - add null pointer protection to MsExtChk logic
 *
 * Revision 1.45  2003/08/22 02:31:38  neal
 * neal - 030821 - enhance MsExtChk filter by adding conservative/aggressive file ext list choice.
 *
 * Revision 1.44  2003/08/21 13:23:44  neal
 * neal - 030821 - add strcasestr for systems without it
 *
 * Revision 1.43  2003/08/20 22:30:09  neal
 * neal - 030820 - add Microsoft file extension attachment vunerablility filter
 *
 * Revision 1.42  2003/08/10 21:53:00  neal
 * neal - 030810 - don\'t cary the sendmail context around to places that only need the private context
 *
 * Revision 1.41  2003/07/06 20:49:00  neal
 * neal - 030706 - fix Exec action for Sender. Add case to handle failure of system.
 *
 * Revision 1.40  2003/07/05 01:01:53  neal
 * neal - 030704 - recipient action now takes precidence over the sender action. add Exec action. remove partially implemented honeypot code in preference of Exec action
 *
 * Revision 1.39  2003/05/28 03:37:57  neal
 * neal - 030527 - include license in all files
 *
 * Revision 1.38  2003/05/14 13:47:45  neal
 * neal - 030514 - do not reject if there was a prior ACCEPT based action
 *
 * Revision 1.37  2003/05/04 06:20:47  neal
 * neal - 030504 - do some cleanup before release
 *
 * Revision 1.36  2003/05/02 02:48:06  neal
 * neal - 030501 - add TarPit to bwlist action types
 *
 * Revision 1.35  2003/05/02 02:39:33  neal
 * neal - 030501 - remove/cleanup debug junk. if sender validation MTA returns 4xx then return TEMPFAIL to sendmail
 *
 * Revision 1.34  2003/05/01 02:31:35  neal
 * neal - 030430 - better smtp send delivery test handling
 *
 * Revision 1.33  2003/04/25 01:17:57  neal
 * neal - 030424 - always record recipient
 *
 * Revision 1.32  2003/04/24 04:11:02  neal
 * neal - 030423 - move bwlist reject from envfrom to envrcpt for logging purposes
 *
 * Revision 1.31  2003/04/17 05:33:34  neal
 * neal - 030417 - fix bwlist action Accept logic in envrcpt
 *
 * Revision 1.30  2003/04/17 05:08:35  neal
 * neal - 030417 - move sender delivery rejection into envrcpt
 *
 * Revision 1.29  2003/04/14 03:52:07  neal
 * neal - 030413 - move the rdnsbl hosts to a flat ascii file
 *
 * Revision 1.28  2003/04/14 02:03:06  neal
 * neal - 030413 - make MTA hostname checking optional
 *
 * Revision 1.27  2003/04/14 01:08:38  neal
 * neal - 030413 - move black and white lists to flat ascii files from sql.
 *
 * Revision 1.26  2003/04/08 14:10:02  neal
 * neal - 030408 - move the MTA hostname rejection to envrcpt so that the recipient can be logged.
 *
 * Revision 1.25  2003/04/08 03:40:54  neal
 * neal - 030407 - add a url for Invalid MTA contitions
 *
 * Revision 1.24  2003/04/03 20:37:13  neal
 * neal - 030403 - add logic to require Helo handshake or reject the connection
 *
 * Revision 1.23  2003/04/03 03:13:53  neal
 * neal - 030402 - move MTA hostname check into default sndr action so that a BWL_A_xxxx will work as expected
 *
 * Revision 1.22  2003/04/01 14:10:12  neal
 * neal - 030401 - if the sender action is ACCEPT, dont flag as spam
 *
 * Revision 1.21  2003/03/31 05:49:59  neal
 * neal - 030330 - do not link the milter to the bind resolver library, it segfaults (and is is huge!)
 *
 * Revision 1.20  2003/03/31 04:39:50  neal
 * neal - 030330 - add MTA hostname validation
 *
 * Revision 1.19  2003/03/30 05:56:02  neal
 * neal - 030329 - move recipient verifycation logging to abort stage so that multiple verifications will be caught
 *
 * Revision 1.18  2003/03/30 05:45:59  neal
 * neal - 030329 - add recipient verifycation logging
 *
 * Revision 1.17  2003/03/30 05:22:11  neal
 * neal - 030329 - make it compile
 *
 * Revision 1.16  2003/03/30 05:21:10  neal
 * neal - 030329 - do sender fail/discard in envrcpt so that we can log the recipient
 *
 * Revision 1.15  2003/03/30 05:04:48  neal
 * neal - 030329 - add logic to handle DISCARD and TEMPFAIL black and white list actions
 *
 * Revision 1.14  2003/03/27 14:34:11  neal
 * neal - 030327 - do not rdnsbl if the sender is whitelisted.
 *
 * Revision 1.13  2003/03/24 04:13:14  neal
 * neal - 030323 - more bad pointer fixes, etc
 *
 * Revision 1.12  2003/03/24 03:46:08  neal
 * neal - 030323 - more bad pointer fixes, etc
 *
 * Revision 1.11  2003/03/24 02:23:09  neal
 * neal - 030323 - bad pointer fixes, etc
 *
 * Revision 1.10  2003/03/24 00:55:26  neal
 * neal - 030323 - fix dnsblchk so that it works
 *
 * Revision 1.9  2003/03/24 00:46:26  neal
 * neal - 030323 - add realtime black/white list of sender/recipient email addresses
 *
 * Revision 1.8  2003/03/20 07:03:22  neal
 * neal - 030320 - do not bother with a sender addres delivery check if the host is already on a DNSBL.
 *
 * Revision 1.7  2003/03/20 06:42:24  neal
 * neal - 030320 - add accept/reject/spam logging. make "Sender address deliverable" check work for local domain.
 *
 * Revision 1.6  2003/03/02 04:11:49  neal
 * neal - 030301 - add smtp sender deliverablity check
 *
 * Revision 1.5  2003/03/01 18:12:45  neal
 * neal - 030301 - optionalize dnsbl all operations
 *
 * Revision 1.4  2003/02/24 08:16:21  neal
 * neal - 030224 - added dnsbl rejection point to RCPT stage, preferenced over connect stage
 *
 * Revision 1.3  2003/02/24 04:11:33  neal
 * neal - 030223 - add header/footers to all files
 *
 *--------------------------------------------------------------------*/
