/*--------------------------------------------------------------------*
 *
 * Developed by;
 *	Neal Horman - http://www.wanlink.com
 *	Copyright (c) 2002 Neal Horman. All Rights Reserved
 *
 *	Redistribution and use in source and binary forms, with or without
 *	modification, are permitted provided that the following conditions
 *	are met:
 *	1. Redistributions of source code must retain the above copyright
 *	   notice, this list of conditions and the following disclaimer.
 *	2. Redistributions in binary form must reproduce the above copyright
 *	   notice, this list of conditions and the following disclaimer in the
 *	   documentation and/or other materials provided with the distribution.
 *	3. All advertising materials mentioning features or use of this software
 *	   must display the following acknowledgement:
 *	This product includes software developed by Neal Horman.
 *	4. Neither the name Neal Horman nor the names of any contributors
 *	   may be used to endorse or promote products derived from this software
 *	   without specific prior written permission.
 *	
 *	THIS SOFTWARE IS PROVIDED BY NEAL HORMAN AND ANY CONTRIBUTORS ``AS IS'' AND
 *	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *	ARE DISCLAIMED.  IN NO EVENT SHALL NEAL HORMAN OR ANY CONTRIBUTORS BE LIABLE
 *	FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 *	DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *	OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 *	LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *	OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *	SUCH DAMAGE.
 *
 *	CVSID:  $Id: inet.c,v 1.8 2005/07/18 00:55:13 neal Exp $
 *
 * DESCRIPTION:
 *	application:	spamilter
 *	module:		inet.c
 *--------------------------------------------------------------------*/

static char const cvsid[] = "@(#)$Id: inet.c,v 1.8 2005/07/18 00:55:13 neal Exp $";

#include <stdio.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <time.h>
#include <sys/time.h>
#include <errno.h>
#include <string.h>

#ifdef OS_SunOS
#include <sys/filio.h>
#endif

#include "inet.h"

void NetSockInitAddr(struct sockaddr_in *pSock, long ip, short port)
{
	// setup the socket address structure
	memset((void *)pSock,0,sizeof(struct sockaddr_in));
	pSock->sin_family	= AF_INET;
	pSock->sin_port		= port != 0 ? htons(port) : 0;
	pSock->sin_addr.s_addr	= ip == INADDR_ANY ? INADDR_ANY : htonl(ip);
}

void NetSockClose(int *pSd)
{
	if(*pSd != INVALID_SOCKET )
	{	int	rc;

		shutdown(*pSd,SHUT_RDWR);
		while((rc = close(*pSd)) != 0 && (errno == EAGAIN || errno == EINPROGRESS || errno == EALREADY))
			sleep(10);
		*pSd = INVALID_SOCKET;
	}
}

int NetSockBind(int *pSd, int sockType, long ip, short port)
{	int			ok = ((*pSd = socket(AF_INET,sockType,0)) != INVALID_SOCKET);
	struct sockaddr_in	socks;

	if(ok)
	{
		NetSockInitAddr(&socks,ip,port);

		// bind the source port
		ok = (bind(*pSd,(struct sockaddr *)&socks,sizeof(socks)) != SOCKET_ERROR);
	}

	return(ok);
}

int NetSockOptNoLinger(int sd)
{	struct linger	optlinger;

	optlinger.l_onoff = 0;
	optlinger.l_linger = 0;

	return(setsockopt(sd,SOL_SOCKET,SO_LINGER,(char *)&optlinger,sizeof(optlinger)) == 0);
}

int NetSockOpt(int sd, int optname, int optval)
{
	return(setsockopt(sd,SOL_SOCKET,optname,(char *)&optval,sizeof(int)) == 0);
}

int NetSockOpenUdpListen(short port)
{	int			sd;
	int			ok = NetSockBind(&sd,SOCK_DGRAM,INADDR_ANY,port);
	unsigned long		arg_on = 1;

	if(ok)
	{
		// set the socket to non-blocking
		ok = (ioctl(sd,FIONBIO,&arg_on) == 0);

		// set the socket to broadcast
		if(ok)
			ok = NetSockOpt(sd,SO_BROADCAST,1);
	}

	if(!ok)
		NetSockClose(&sd);

	return(sd);
}

int NetSockOpenTcpPeer(long ip, short port)
{	int			sd	= INVALID_SOCKET;
	unsigned long		arg_on	= 1;
	int			ok	= NetSockBind(&sd,SOCK_STREAM,INADDR_ANY,0) && (ioctl(sd,FIONBIO,&arg_on) == 0);
	int			err;
	struct sockaddr_in	socks;

	if(ok)
	{
		NetSockInitAddr(&socks,ip,port);
		ok = NetSockOptNoLinger(sd);

		if(ok)
		{
			if((err = connect(sd,(struct sockaddr *)&socks,sizeof(socks))) == SOCKET_ERROR)
				err = errno;
			ok = (err == 0 || err == EINPROGRESS || err == EWOULDBLOCK);
		}
	}

	if(!ok)
		NetSockClose(&sd);

	return(sd);
}

int NetSockOpenTcpListen(long ip, short port)
{	int			sd,ok;
	unsigned long		arg_on	= 1;
	struct sockaddr_in	socks;

	NetSockInitAddr(&socks,ip,port);
	ok = ((sd = socket(AF_INET,SOCK_STREAM,0)) != INVALID_SOCKET) &&
		NetSockOptNoLinger(sd) &&
		NetSockOpt(sd,SO_REUSEADDR,1) &&
		(ioctl(sd,FIONBIO,&arg_on) == 0) &&
		(bind(sd,(struct sockaddr *)&socks,sizeof(socks)) != SOCKET_ERROR) &&
		(listen(sd,1) != SOCKET_ERROR);

	if(!ok)
		NetSockClose(&sd);

	return(sd);
}

/* buffer up lines of data from peer until 1 second time-out */
int NetSockGets(int sd, char *buf, int buflenmax, int timeout)
{	char		*str = buf;
	fd_set		sfds;
	struct timeval	tv;
	int		rc = 0;
	int		done = 0;

	/* reset buffer for next iteration */
	memset(buf,0,buflenmax);
	while(!done)
	{

		/* our own fgets... cept for handles */
		while((rc = read(sd,str,1)) == 1 && *str != '\n' && str < buf+buflenmax-1)
			str++;

		/* if we have a complete line, clean it up, and handle it */
		if(rc == 1 && strlen(buf) && *str=='\n')
		{
			/* trim right */
			while((*str =='\r' || *str == '\n') && str > buf)
				*(str--) = '\0';
			/* trim left */
			str = buf;
			while(*str==' ' || *str =='\t' || *str=='\r' || *str=='\n')
				str++;
			/* handle complete line by breaking loop */
			if(*str)
			{	char *dst = buf;

				/* left justify the buffer */
				if(str>dst)
				{
					while(*str)
						*(dst++) = *(str++);
					*dst = '\0';
				}
				done = 1;
				rc = strlen(buf);
			}
			else
			{
				done = 1;
				rc = 0;
			}
		}
		else if(rc == 0)
			done = 1;
		else	/* try to wait for more */
		{
			tv.tv_sec = timeout;
			tv.tv_usec = 0;
			FD_ZERO(&sfds);
			FD_SET(sd,&sfds);
			rc = select(FD_SETSIZE,&sfds,NULL,NULL,&tv);
			done = (rc == 0 || rc == -1);
			if(done)
				rc = -1;
		}
	}

	return(rc);
}

int NetSockVPrintf(int sd, char *fmt, va_list vl)
{	char	*str;
	int	rc = -1;

	if(sd != INVALID_SOCKET)
	{
		rc = vasprintf(&str,fmt,vl);
		if(str != NULL && rc != -1)
			rc = send(sd,str,rc,0);
		if(str != NULL)
			free(str);
	}

	return(rc);
}

int NetSockPrintf(int sd, char *fmt, ...)
{	va_list	vl;
	int	rc;

	va_start(vl,fmt);
	rc = NetSockVPrintf(sd,fmt,vl);
	va_end(vl);

	return(rc);
}

/*--------------------------------------------------------------------*
 * $Log: inet.c,v $
 * Revision 1.8  2005/07/18 00:55:13  neal
 * neal - 050717 - Wall cleanup
 *
 * Revision 1.7  2004/08/18 01:55:31  neal
 * neal - 040817 - add NetSockVPrintf
 *
 * Revision 1.6  2004/03/23 14:02:16  neal
 * neal - 040323 - really close non-blocking sockets
 *
 * Revision 1.5  2004/01/05 02:32:50  neal
 * neal - 040104 - add preliminary compile suport for solaris
 *
 * Revision 1.4  2003/09/02 07:12:10  neal
 * neal - 030902 - fix segfaults. add MtaHostIpfw filter
 *
 * Revision 1.3  2003/05/28 03:37:57  neal
 * neal - 030527 - include license in all files
 *
 * Revision 1.2  2003/03/24 00:46:26  neal
 * neal - 030323 - add realtime black/white list of sender/recipient email addresses
 *
 * Revision 1.1  2003/03/02 04:11:49  neal
 * neal - 030301 - add smtp sender deliverablity check
 *
 *--------------------------------------------------------------------*/
